/*
 * Copyright (c) 2001 Jilles van Gurp
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package newstate;

import java.util.*;

/**
* This class serves as a access point for the whole framework. An FSM object 
* encapsulates a FSM and provides a factory method to create a FSMContext 
* object for this FSM.
*/
public class FSM implements java.io.Serializable
{
        Hashtable states = new Hashtable();
        Vector events = new Vector();
        State first = null;
        FSMAction initaction = null;
        
        /**
        * This method can be used to add a state to the FSM.
        * FSM uses the State class to create a new State object
        * @param statename The name of the state
        */
        public void addState(String statename)
        {
                if(!states.containsKey(statename))
                {
                        State s = new State(statename);                
                        states.put(statename, s);
                }
                else throw new RuntimeException("state; " + statename + 
                                                " already declared");
        }                

        /**
        * This method can be used to add a state to the FSM.
        * FSM uses the State class to create a new State object
        * @param entryAction The action that is executed upon state-entry
        * @param statename The name of the state
        */
        public void addState(FSMAction entryAction, String statename)
        {
                if(!states.containsKey(statename))
                {
                        State s = new State(statename);
                        s.setStateEntryAction(entryAction);
                        states.put(statename, s);
                }
                else throw new RuntimeException("state; " + statename + 
                                                " already declared");
        }                

        /**
        * This method can be used to add a state to the FSM.
        * FSM uses the State class to create a new State object
        * @param statename The name of the state
        * @param exitAction The action that is executed upon state-exit
        */
        public void addState(String statename, FSMAction exitAction)
        {
                if(!states.containsKey(statename))
                {
                        State s = new State(statename);
                        s.setStateExitAction(exitAction);
                        states.put(statename, s);
                }
                else throw new RuntimeException("state; " + statename + 
                                                " already declared");
        }                

        /**
        * This method can be used to add a state to the FSM.
        * FSM uses the State class to create a new State object
        * @param entryAction The action that is executed upon state-entry
        * @param statename The name of the state
        * @param exitAction The action that is executed upon state-exit
        */
        public void addState(FSMAction entryAction, String statename, 
                             FSMAction exitAction)
        {
                if(!states.containsKey(statename))
                {
                        State s = new State(statename);
                        s.setStateEntryAction(entryAction);
                        s.setStateExitAction(exitAction);
                        states.put(statename, s);
                }
                else throw new RuntimeException("state; " + statename + 
                                                " already declared");
        }                

        /**
        * This method can be used to add an event to the FSM.
        * FSM uses the Event class to create a new Event object
        * @param name This is the name of the event.
        */
        public void addEvent(String name)
        {
                if(!events.contains(name))
                {
                        events.addElement(name);
                }
                else throw new RuntimeException("event; " + name + 
                                                " already declared");
        }
        
        /**
        * This method creates a transition between the sourcestate and the
        * target state. The method checks whether the given states and the
        * event exist before it creates the transition. If they don't exist
        * a RuntimeException is thrown.
        * @param sourcestate The name of the sourcestate
        * @param eventname The name of the event that triggers the transition
        * @param targetstate the name of the targetstate
        * @param action The action that will be executed when the transition
        * is triggered.
        */
        public void addTransition(String sourcestate, String eventname, 
                                  String targetstate, FSMAction action)
        {
                if(states.containsKey(sourcestate))
                {
                        State s = (State)states.get(sourcestate);
                        if(states.containsKey(targetstate))
                        {
                                State t = (State)states.get(targetstate);
                                if(events.contains(eventname))
                                {
                                        s.addTransition(eventname, t, action);
                                }
                                else throw new RuntimeException("event; " + 
                                      eventname + " not found");
                        }
                        else throw new RuntimeException("state; " + targetstate
                               + " not found");
                }
                else throw new RuntimeException("state; " + sourcestate + 
                                                " not found");
        }                

        /**
        * This method is used to set the default state for the FSM. If a 
        * FSMContext is created, this state is set as the current state.
        * @param statename The name of the first action.
        */
        public void setFirstState(String statename)
        {
                first = (State)states.get(statename);
        }

        /**
        * Sometimes it's necessary to do some initialization before the FSM can
        * be used. For this purpos a initial action can be set. This action is
        * executed when the FSMContext is created. 
        * @param action The initial action.
        */
        public void setInitAction(FSMAction action)
        {
                initaction = action;
        }
       
        /**
        * This method serves as a factory method to create FSMContexts from
        * the FSM. Also the init action is run (if available).
        * @return A new FSMContext for the FSM.
        */
        public FSMContext createFSMInstance()
        {
                FSMContext fsmc;
                if(first == null) throw new Error("first state not set");
                else
                {
                        if(initaction != null)
                        {
                                fsmc =  new FSMContext(first, this, initaction);
                        }
                        else
                        {
                                fsmc =  new FSMContext(first, this);
                        }
                        return fsmc;
                }
        }
}
