/*
 * Copyright (c) 2001 Jilles van Gurp
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package newstate;

import java.util.*;

/**
* This is the context of a FSM. A context holds a reference to the current state
* and also functions as a repository for objects. To do that it extends from
* java.util.Hashtable. Objects can be stored with the hashtable's put method and
* retrieved with the hashtables get method.
*/
public class FSMContext extends Hashtable implements java.io.Serializable
{
        private State state;
        private State firstState;
        private FSMAction initialAction = State.skip;
        private FSM theFSM;
                
        public FSMContext()
        {
        }
        
        /**
        * Create a new context with s as the first state
        * @param s The first state
        */
        public FSMContext(State s, FSM fsm)
        {
                super();
                setFirstState(s);
                setFSM(fsm);
                initialize();
        }
        
        /**
        * Create a new context with s as the first state. an then execute i to
        * initialize the context.
        * @param s The first state
        * @param i The initial action
        */
        public FSMContext(State s, FSM fsm, FSMAction i)
        {
                super();
                setFirstState(s);
                setInitialAction(i);
                setFSM(fsm);
                initialize();
        }
        
        public void setFSM(FSM fsm) {theFSM=fsm;}
        public FSM getFSM() {return theFSM;}
        /**
        * Find out what events can be sent to the current state
        * @return A list of events
        */
        public Vector getEvents()
        {
                return getState().getEvents();
        }
               
        
        /**
        * Initialize the context. Set first state and execute initial action.
        */
        public void initialize()
        {
                this.clear();
                initialAction.execute(this, null);
                setState(firstState);
                firstState.getStateEntryAction().execute(this, null);
        }
        
        public void setState(State s) { state = s; }
        public State getState() { return state; }
        
        public void setFirstState(State s) { firstState = s; }
        public State getFirstState() { return firstState; }
        
        public void setInitialAction(FSMAction a) { initialAction = a; }
        public FSMAction getInitialAction() { return initialAction; }
        
        /**
        * Dispatch an event e.
        * @param e The event
        - @param data Some additional data
        */
        public void dispatch(String event, Object data)
        {
                getState().dispatch(event, data, this);
        }
}
