/*
 * Copyright (c) 2001 Jilles van Gurp
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package newstate;

import java.util.*;
import newstate.tools.*;

/**
* This class models a State. A state has a name and entry /exit actions.
* Further state also contains a dispatch mechanism for incoming events.
*/
class State implements java.io.Serializable
{
        private String name;
        private Hashtable transitions = new Hashtable();
        
        public static SkipAction skip = new SkipAction();
        
        FSMAction onStateEntry = skip;
        FSMAction onStateExit = skip;

        /**
        * Initilizes a stateobject with name s.
        * @param s The name of the new state
        */
        public State(String s) 
        { 
                setName(s); 
        }
        
        public void setName(String s) { name = s; }
        public String getName() { return name; }
        
        public void setStateEntryAction(FSMAction action) 
        { 
                onStateEntry = action; 
        }
        public FSMAction getStateEntryAction() 
        { 
                return onStateEntry; 
        }

        public void setStateExitAction(FSMAction action) 
        {
                onStateExit = action; 
        }
        
        public FSMAction getStateExitAction() 
        {
                return onStateExit;
        }
        
        /**
        * Convenience method that returns the name of this state.
        */
        public String toString()
        {
                return getName();
        }
        
        /**
        * Adds a transition with this state as source and parameter to as a
        * target. 
        * @param trigger The event that triggers the transition
        * @param to The target state.
        * @param action The associated action
        */
        public void addTransition(String trigger, State to, FSMAction action)
        {
                transitions.put(trigger, new Transition(this,to,action));
        }

        /**
        * Dispatch an event.
        * @param trigger The event that needs to be dispatched. The correct 
        * transition is located and than executed.
        * @param data Some additional data that may be needed by the action
        * @param fsmc The context in which the action is executed. This may be 
        * useful for retrieving global variables.
        */
        public void dispatch(String trigger, Object data, FSMContext fsmc)
        {
                ((Transition)transitions.get(trigger)).execute(fsmc, data);
        }
        
        /**
        * Method to find out which events can be dispatched by this state.
        * @return A vector with the events 
        */
        public Vector getEvents()
        {
                Vector v = new Vector();
                for(Enumeration e = transitions.keys() ; e.hasMoreElements();)
                {
                        v.addElement(e.nextElement());
                }
                return v;
        }
}
